<?php

/*
 * OAuth2 Client Bundle
 * Copyright (c) KnpUniversity <http://knpuniversity.com/>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace KnpU\OAuth2ClientBundle\tests\Client;

use KnpU\OAuth2ClientBundle\Client\OAuth2PKCEClient;
use League\OAuth2\Client\Provider\AbstractProvider;
use League\OAuth2\Client\Provider\ResourceOwnerInterface;
use League\OAuth2\Client\Token\AccessToken;
use PHPUnit\Framework\TestCase;
use Psr\Http\Message\ResponseInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpFoundation\Session\Session;
use Symfony\Component\HttpFoundation\Session\Storage\MockArraySessionStorage;

class OAuth2PKCEClientTest extends TestCase
{
    private RequestStack $requestStack;
    private Request $request;
    private Session $session;
    private AbstractProvider $provider;

    public function setup(): void
    {
        $this->session = new Session(new MockArraySessionStorage());
        $this->request = new Request();
        $this->request->setSession($this->session);
        $this->requestStack = new RequestStack();
        $this->requestStack->push($this->request);
        $this->provider = new class extends AbstractProvider {

            protected function checkResponse(ResponseInterface $response, $data): void
            {
                // not needed
            }

            protected function createResourceOwner(array $response,
                                                   AccessToken $token): ResourceOwnerInterface
            {
                // not needed
            }

            protected function getDefaultScopes(): array
            {
                return [];
            }

            public function getBaseAccessTokenUrl(array $params): string
            {
                // not needed
            }

            public function getBaseAuthorizationUrl(): string
            {
                return 'http://coolOAuthServer.com/authorize';
            }

            public function getResourceOwnerDetailsUrl(AccessToken $token): string
            {
                // not needed
            }

            public function getAccessToken($grant, array $options = [])
            {
                // hijack this method to just create a new AccessToken using the $options
                return new AccessToken($options + ['access_token' => 'DUMMY_ACCESS_TOKEN']);
            }
        };
    }

    public function testRedirect()
    {
        $client = new OAuth2PKCEClient(
            $this->provider,
            $this->requestStack
        );
        $client->setAsStateless();

        $response = $client->redirect();
        $this->assertInstanceOf(
            RedirectResponse::class,
            $response
        );

        // assert the code_challenge & code_challenge_method parameters are in the redirect response
        $queries = [];
        parse_str(parse_url($response->getTargetUrl(), PHP_URL_QUERY), $queries);
        $this->assertArrayHasKey('code_challenge', $queries);
        $this->assertArrayHasKey('code_challenge_method', $queries);
    }

    public function testGetAccessToken()
    {
        $client = new OAuth2PKCEClient(
            $this->provider,
            $this->requestStack
        );
        // skip state checking
        $client->setAsStateless();

        // ensure code verifier is generated by redirect() and stored in session first
        $client->redirect();
        // OAuth2Client::getAccessToken() requires 'code' query parameter
        $this->request->query->set('code', 'DUMMY_CODE');

        // assert the code_verifier parameter is passed and returned back by the hijacked provider
        $accessToken = $client->getAccessToken();
        $this->assertArrayHasKey('code_verifier', $accessToken->getValues());
    }
}
