<?php

namespace App\Service\Layout\Widget\Page\OpenGraph\Article;

use App\Service\Layout\Compiler\Context\CompilerContext;
use App\Service\Layout\Compiler\Exception\StampNotFoundException;
use App\Service\Layout\Compiler\Node;
use App\Service\Layout\Compiler\Stamp\OpenGraphArticleStamp;
use App\Service\Layout\Compiler\StampProcessorInterface;
use App\Service\Layout\Widget\Page\PageProperties;

final readonly class ArticleStampProcessor implements StampProcessorInterface
{
    public function supports(Node $node, CompilerContext $context): bool
    {
        return $node->getType() === 'page'
            && $node->getProperties() instanceof PageProperties
            && $context->hasStamp(OpenGraphArticleStamp::class);
    }

    /**
     * @throws StampNotFoundException
     */
    public function process(Node $node, CompilerContext $context): Node
    {
        /** @var OpenGraphArticleStamp $stamp */
        $stamp = $context->getHighestPriorityStamp(OpenGraphArticleStamp::class);

        $author = null;
        $authorExists = $stamp->getAuthorUsername() !== null
            || ($stamp->getAuthorFirstName() !== null && $stamp->getAuthorLastName() !== null);

        if ($authorExists) {
            $author = new Author(
                $stamp->getAuthorFirstName(),
                $stamp->getAuthorLastName(),
                $stamp->getAuthorUsername(),
                null
            );
        }

        $article = new Article(
            $stamp->getPublishedTime(),
            $stamp->getModifiedTime(),
            $stamp->getExpirationTime(),
            $author,
            $stamp->getSection(),
            $stamp->getTag()
        );

        /** @var PageProperties $properties */
        $properties = $node->getProperties();

        $openGraph = $properties->getOpenGraph();
        $openGraph = $openGraph->setArticle($article);

        $properties = $properties->setOpenGraph($openGraph);

        return $node->setProperties($properties);
    }
}