<?php

namespace App\Service\Layout\Compiler;

use App\Service\Layout\Compiler\Context\CompilerContext;
use App\Service\Layout\Compiler\Event\NodeCompiledEvent;
use App\Service\Layout\Compiler\Event\PreAddChildEvent;
use App\Service\Layout\Compiler\Event\PreCompileNodeEvent;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

final readonly class RecursiveCompiler implements CompilerInterface
{
    /**
     * @param EventDispatcherInterface $eventDispatcher
     * @param iterable<NodeCompilerInterface> $nodeCompilers
     * @param iterable<StampProcessorInterface> $stampProcessors
     */
    public function __construct(
        private EventDispatcherInterface $eventDispatcher,
        private iterable                 $nodeCompilers,
        private iterable                 $stampProcessors
    )
    {
    }

    public function compile(Node $node, CompilerContext $context): Node
    {
        $node = $this->traverse($node, $context);
        $node = $this->processStamps($node, $context);

        return $node;
    }

    private function traverse(Node $node, CompilerContext $context): Node
    {
        $preCompileEvent = new PreCompileNodeEvent($node, $context);
        $this->eventDispatcher->dispatch($preCompileEvent);

        if ($preCompileEvent->isCancelled()) {
            return new NullNode();
        }

        $node = $preCompileEvent->getNode();
        $node = $this->compileNode($node, $context);

        $newChildren = [];

        foreach ($node->getChildren() as $childNode) {
            $childNode = $this->traverse($childNode, $context);

            $preAddChildEvent = new PreAddChildEvent($node, $childNode, $context);
            $this->eventDispatcher->dispatch($preAddChildEvent);

            if (!$preAddChildEvent->isCancelled()) {
                $childNode = $preAddChildEvent->getChild();

                $newChildren[] = $childNode;
            }
        }

        $node = $node->setChildren($newChildren);

        $this->eventDispatcher->dispatch(new NodeCompiledEvent($node, $context));

        return $node;
    }

    private function compileNode(Node $node, CompilerContext $context): Node
    {
        foreach ($this->nodeCompilers as $nodeCompiler) {
            if ($nodeCompiler->supports($node, $context)) {
                return $nodeCompiler->compile($node, $context);
            }
        }

        return $node;
    }

    private function processStamps(Node $node, CompilerContext $context): Node
    {
        foreach ($this->stampProcessors as $stampProcessor) {
            if ($stampProcessor->supports($node, $context)) {
                $node = $stampProcessor->process($node, $context);
            }
        }

        $newChildren = [];

        foreach ($node->getChildren() as $child) {
            $newChild = $this->processStamps($child, $context);

            $newChildren[] = $newChild;
        }

        $node = $node->setChildren($newChildren);

        return $node;
    }
}