<?php

namespace App\Service\Chart;

use DateTime;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\Persistence\ObjectRepository;
use Symfony\Contracts\Translation\TranslatorInterface;
use Symfony\UX\Chartjs\Builder\ChartBuilderInterface;
use Symfony\UX\Chartjs\Model\Chart;

class BaseChart
{
    public const KEY = '';

    public const ENTITY = '';

    public const OPTIONS = [
        'color' => 'rgb(29, 55, 80)',
        'secondary_color' => 'rgb(254, 138, 77)'
    ];

    public readonly ObjectRepository $repository;

    public readonly array $options;

    public function __construct(
        private readonly ChartBuilderInterface  $chartBuilder,
        public readonly TranslatorInterface     $translator,
        private readonly EntityManagerInterface $entityManager
    )
    {
        $this->repository = $this->entityManager->getRepository(static::ENTITY);
        $this->options = static::OPTIONS;
    }

    public function init(array $datasets, string $type): Chart
    {
        $chart = $this->chartBuilder->createChart($type);
        $processed = $this->transformDataSets($datasets);

        $chart->setData([
            'datasets' => $processed['datasets']
        ]);

        $chart->setOptions([
            'scales' => [
                'y' => [
                    'ticks' => [
                        'precision' => 0,
                        'fontFamily' => 'Open Sans, sans-serif',
                        'fontSize' => 12,
                        'fontcolor' => '#97a4af',
                    ],
                    'suggestedMin' => 0,
                    'suggestedMax' => $processed['suggestedMax'] * 2,
                ],
            ],
        ]);

        $chart->setAttributes([
            'id' => static::KEY . 'Chart' . random_int(0, 9999),
            'total' => $processed['total'],
        ]);

        return $chart;

    }

    /**
     * @return array{datasets?: array<int, array{label: mixed, backgroundColor: mixed, borderColor: mixed, parsing: array{yAxisKey: string}, data: mixed}>&mixed[], suggestedMax: mixed, total: float|int}
     */
    private function transformDataSets(array $datasets): array
    {
        $result = [
            'total' => 0,
            'suggestedMax' => 0,
        ];

        foreach ($datasets as $dataset) {
            $data = array_column($dataset['data'], 'count');

            foreach ($dataset['data'] as &$datasetItem) {
                $month = $datasetItem['x'] ?? '2023-01';
                $datasetItem['x'] = (new DateTime($month . '-01'))->format('F');
            }

            $result['total'] += array_sum($data);
            $result['suggestedMax'] = ($data !== [] && $result['suggestedMax'] < max($data)) ? max($data) : $result['suggestedMax'];

            $result['datasets'][] = [
                'label' => $dataset['label'] ?? $this->translator->trans(static::KEY, [], 'cms'),
                'backgroundColor' => $dataset['backgroundColor'] ?? $this->options['color'],
                'borderColor' => $dataset['borderColor'] ?? $this->options['color'],
                'parsing' => ['yAxisKey' => 'count'],
                'data' => $dataset['data'],
            ];
        }

        return $result;
    }
}
