<?php

declare(strict_types=1);

namespace App\Security\Authenticator\Portal;

use App\DTO\PublicAPI\User\Input\FormLogin;
use App\Exception\View\ConstraintViolationException;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Security\Core\Authentication\Token\TokenInterface;
use Symfony\Component\Security\Core\Authentication\Token\UsernamePasswordToken;
use Symfony\Component\Security\Core\Exception\AuthenticationException;
use Symfony\Component\Security\Core\User\PasswordUpgraderInterface;
use Symfony\Component\Security\Core\User\UserProviderInterface;
use Symfony\Component\Security\Http\Authentication\AuthenticationFailureHandlerInterface;
use Symfony\Component\Security\Http\Authentication\AuthenticationSuccessHandlerInterface;
use Symfony\Component\Security\Http\Authenticator\AbstractAuthenticator;
use Symfony\Component\Security\Http\Authenticator\Passport\Badge\PasswordUpgradeBadge;
use Symfony\Component\Security\Http\Authenticator\Passport\Badge\UserBadge;
use Symfony\Component\Security\Http\Authenticator\Passport\Credentials\PasswordCredentials;
use Symfony\Component\Security\Http\Authenticator\Passport\Passport;
use Symfony\Component\Serializer\SerializerInterface;
use Symfony\Component\Validator\Validator\ValidatorInterface;

final class PasswordAuthenticator extends AbstractAuthenticator
{
    public function __construct(
        private readonly SerializerInterface          $serializer,
        private readonly ValidatorInterface           $validator,
        private readonly UserProviderInterface        $userProvider,
        private readonly AuthenticationSuccessHandlerInterface $authenticationSuccessHandler,
        private readonly AuthenticationFailureHandlerInterface $authenticationFailureHandler
    )
    {
    }

    public function supports(Request $request): ?bool
    {
        return $request->attributes->get('_route') === 'portal_api_auth_login_password_verify';
    }

    public function authenticate(Request $request): Passport
    {
        $formLogin = $this->getCredentials($request);

        $userBadge = new UserBadge($formLogin->username, $this->userProvider->loadUserByIdentifier(...));
        $passport = new Passport($userBadge, new PasswordCredentials($formLogin->password));

        if ($this->userProvider instanceof PasswordUpgraderInterface) {
            $passport->addBadge(new PasswordUpgradeBadge($formLogin->password, $this->userProvider));
        }

        return $passport;
    }

    private function getCredentials(Request $request): FormLogin
    {
        $formLogin = $this->serializer->deserialize($request->getContent(), FormLogin::class, 'json');

        $constraintViolationList = $this->validator->validate($formLogin);

        if ($constraintViolationList->count() > 0) {
            throw new ConstraintViolationException($constraintViolationList);
        }

        return $formLogin;
    }

    public function createToken(Passport $passport, string $firewallName): TokenInterface
    {
        return new UsernamePasswordToken($passport->getUser(), $firewallName, $passport->getUser()->getRoles());
    }

    public function onAuthenticationSuccess(Request $request, TokenInterface $token, string $firewallName): ?Response
    {
        return $this->authenticationSuccessHandler->onAuthenticationSuccess($request, $token);
    }

    public function onAuthenticationFailure(Request $request, AuthenticationException $exception): ?Response
    {
        return $this->authenticationFailureHandler->onAuthenticationFailure($request, $exception);
    }
}
