<?php

declare(strict_types=1);

namespace App\Form\System\Administrator;

use App\Entity\System\Administrator\Administrator;
use App\Enum\Shared\TwoFactorProvider;
use Symfony\Component\DependencyInjection\ParameterBag\ParameterBagInterface;
use Symfony\Component\Form\AbstractType;
use Symfony\Component\Form\Extension\Core\Type\ChoiceType;
use Symfony\Component\Form\Extension\Core\Type\EmailType;
use Symfony\Component\Form\Extension\Core\Type\EnumType;
use Symfony\Component\Form\Extension\Core\Type\FormType;
use Symfony\Component\Form\Extension\Core\Type\LanguageType;
use Symfony\Component\Form\Extension\Core\Type\PasswordType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\Form\FormEvent;
use Symfony\Component\Form\FormEvents;
use Symfony\Component\OptionsResolver\OptionsResolver;

final class ManageType extends AbstractType
{
    public function __construct(
        private readonly ParameterBagInterface $parameterBag,
    )
    {
    }

    public function buildForm(FormBuilderInterface $builder, array $options): void
    {
        $builder
            ->add($builder->create('general', FormType::class, ['inherit_data' => true]))
            ->add($builder->create('relations', FormType::class, ['inherit_data' => true]));

        $builder
            ->get('general')
            ->add('firstName')
            ->add('lastName')
            ->add('email', EmailType::class);

        $builder->get('relations')
            ->add('roles', ChoiceType::class, [
                'required' => true,
                'multiple' => true,
                'autocomplete' => true,
                'choices' => [
                    'role_user' => 'ROLE_USER',
                    'role_editor' => 'ROLE_PARTNER',
                    'role_admin' => 'ROLE_ADMIN',
                    'role_super_admin' => 'ROLE_SUPER_ADMIN',
                ],
            ])
            ->add('locale', LanguageType::class, [
                'choices' => $this->parameterBag->get('enabled_locales'),
                'preferred_choices' => [$this->parameterBag->get('default_locale')],
            ])
            ->add('twoFactorProvider', EnumType::class, [
                'class' => TwoFactorProvider::class,
                'required' => false
            ]);

        $builder->addEventListener(FormEvents::PRE_SUBMIT, function (FormEvent $event) {
            $existAdministrator = (bool)$event->getForm()->getData()->getId();
            $form = $event->getForm();

            if (!$existAdministrator) {
                $form['general']->add('plainPassword', PasswordType::class, [
                    'hash_property_path' => 'password',
                    'mapped' => false,
                    'required' => true,
                    'empty_data' => bin2hex(random_bytes(12))
                ]);
            }
        });
    }

    public function configureOptions(OptionsResolver $resolver): void
    {
        $resolver->setDefaults([
            'data_class' => Administrator::class,
            'locale' => ''
        ]);
    }
}
