<?php

namespace App\EventSubscriber\System;

use App\Entity\Interfaces\NestedableInterface;
use Doctrine\Bundle\DoctrineBundle\EventSubscriber\EventSubscriberInterface;
use Doctrine\ORM\Event\LoadClassMetadataEventArgs;
use Doctrine\ORM\Events;
use Doctrine\ORM\Mapping\ClassMetadataInfo;
use ReflectionClass;

final class NestedableEventSubscriber implements EventSubscriberInterface
{
    private const ROOT = 'root';

    private const PARENT = 'parent';

    private const CHILDREN = 'children';

    public function loadClassMetadata(LoadClassMetadataEventArgs $loadClassMetadataEventArgs): void
    {
        $classMetadata = $loadClassMetadataEventArgs->getClassMetadata();
        $reflectionClass = $classMetadata->getReflectionClass();

        if (!$reflectionClass instanceof ReflectionClass) {
            return;
        }

        if (!$reflectionClass->implementsInterface(NestedableInterface::class)) {
            return;
        }

        $this->mapNestedableFields($classMetadata);
    }

    private function mapNestedableFields(ClassMetadataInfo $classMetadataInfo): void
    {
        if (!$classMetadataInfo->hasField(self::ROOT)) {
            $classMetadataInfo->mapManyToOne([
                'fieldName' => self::ROOT,
                'targetEntity' => $classMetadataInfo->getName(),
                'inversedBy' => null,
                'joinColumns' => [
                    [
                        'name' => 'tree_root',
                        'referencedColumnName' => 'id',
                        'onDelete' => 'CASCADE',
                    ],
                ],
            ]);
        }

        if (!$classMetadataInfo->hasField(self::PARENT)) {
            $classMetadataInfo->mapManyToOne([
                'fieldName' => self::PARENT,
                'targetEntity' => $classMetadataInfo->getName(),
                'inversedBy' => self::CHILDREN,
                'joinColumns' => [
                    [
                        'name' => 'parent_id',
                        'referencedColumnName' => 'id',
                        'onDelete' => 'CASCADE',
                    ],
                ],
            ]);
        }

        if (!$classMetadataInfo->hasField(self::CHILDREN)) {
            $classMetadataInfo->mapOneToMany([
                'fieldName' => self::CHILDREN,
                'targetEntity' => $classMetadataInfo->getName(),
                'mappedBy' => self::PARENT,
                'orderBy' => [
                    'lft' => 'ASC',
                ],
                'cascade' => [
                    'persist',
                    'remove',
                ],
            ]);
        }
    }

    public function getSubscribedEvents(): array
    {
        return [
            Events::loadClassMetadata,
        ];
    }
}
