<?php

declare(strict_types=1);

namespace App\EventSubscriber\Portal\Email;

use App\Event\User\UserPasswordResetEvent;
use App\Service\Url\PortalUrlGeneratorInterface;
use Symfony\Bridge\Twig\Mime\TemplatedEmail;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\Mailer\MailerInterface;
use Symfony\Contracts\Translation\TranslatorInterface;

final readonly class UserPasswordResetEmailEventSubscriber implements EventSubscriberInterface
{
    public function __construct(
        private PortalUrlGeneratorInterface $portalUrlGenerator,
        private TranslatorInterface         $translator,
        private MailerInterface             $mailer
    )
    {
    }

    public static function getSubscribedEvents(): array
    {
        return [
            UserPasswordResetEvent::class => 'handle'
        ];
    }

    public function handle(UserPasswordResetEvent $userPasswordResetEvent): void
    {
        $token = $userPasswordResetEvent->getToken()->getToken();
        $link = $this->portalUrlGenerator->generateUserPasswordResetUrl($token);

        $templatedEmail = (new TemplatedEmail())
            ->to($userPasswordResetEvent->getUser()->getEmail())
            ->subject($this->translator->trans('New password request'))
            ->htmlTemplate('email/user_password_reset.html.twig')
            ->context([
                'name' => $userPasswordResetEvent->getUser()->getFirstName(),
                'link' => $link
            ]);
        $this->mailer->send($templatedEmail);
    }
}
