<?php

declare(strict_types=1);

namespace App\Entity\User;

use App\Entity\Interfaces\AddressableInterface;
use App\Entity\Interfaces\VerifiableInterface;
use App\Entity\Shared\AbstractEntity;
use App\Enum\User\UserDeletionReason;
use App\Repository\User\UserRepository;
use App\Traits\Entity\Addressable;
use App\Traits\Entity\NameAware;
use App\Traits\Entity\Verifiable;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Symfony\Component\Security\Core\User\PasswordAuthenticatedUserInterface;
use Symfony\Component\Security\Core\User\UserInterface;
use Symfony\Component\Validator\Constraints as Assert;

#[ORM\Table(name: "portal_user")]
#[ORM\Entity(repositoryClass: UserRepository::class)]
class User extends AbstractEntity implements
    UserInterface,
    PasswordAuthenticatedUserInterface,
    VerifiableInterface,
    AddressableInterface
{
    use Verifiable, Addressable, NameAware;

    #[ORM\Column(length: 255)]
    #[Assert\NotBlank]
    #[Gedmo\Versioned]
    private ?string $email = null;

    #[ORM\Column(length: 255, nullable: true)]
    private ?string $password = null;

    #[ORM\Column]
    private array $roles = [];

    #[ORM\Column(length: 255, nullable: true)]
    private ?string $facebookId = null;

    #[ORM\Column(length: 255, nullable: true)]
    private ?string $googleId = null;

    #[ORM\Column(nullable: true, options: ["default" => null])]
    #[Gedmo\Versioned]
    private ?UserDeletionReason $deletionReason = null;

    #[ORM\Column(nullable: true, options: ["default" => null])]
    #[Gedmo\Versioned]
    private ?string $deletionOtherReason = null;

    #[ORM\Column(type: 'boolean', nullable: false)]
    #[Gedmo\Versioned]
    private bool $verified = false;

    #[ORM\Column(length: 255, nullable: true)]
    private ?string $verificationToken = null;

    #[ORM\Column(length: 3)]
    #[Gedmo\Versioned]
    private ?string $locale = null;

    public function getUserIdentifier(): string
    {
        return $this->email;
    }

    public function getEmail(): ?string
    {
        return $this->email;
    }

    public function setEmail(?string $email): void
    {
        $this->email = $email;
    }

    public function getPassword(): ?string
    {
        return $this->password;
    }

    public function setPassword(?string $password): void
    {
        $this->password = $password;
    }

    public function getFacebookId(): ?string
    {
        return $this->facebookId;
    }

    public function setFacebookId(?string $facebookId): void
    {
        $this->facebookId = $facebookId;
    }

    public function getGoogleId(): ?string
    {
        return $this->googleId;
    }

    public function setGoogleId(?string $googleId): void
    {
        $this->googleId = $googleId;
    }

    public function getDeletionReason(): ?UserDeletionReason
    {
        return $this->deletionReason;
    }

    public function setDeletionReason(?UserDeletionReason $deletionReason): void
    {
        $this->deletionReason = $deletionReason;
    }

    public function getDeletionOtherReason(): ?string
    {
        return $this->deletionOtherReason;
    }

    public function setDeletionOtherReason(?string $deletionOtherReason): void
    {
        $this->deletionOtherReason = $deletionOtherReason;
    }

    public function getRoles(): array
    {
        $roles = $this->roles;
        $roles[] = 'ROLE_USER';

        return array_unique($roles);
    }

    public function setRoles(array $roles): void
    {
        $this->roles = $roles;
    }

    public function eraseCredentials(): void
    {
    }


    public function isVerified(): bool
    {
        return $this->verified;
    }

    public function setVerified(bool $verified): void
    {
        $this->verified = $verified;
    }

    public function getVerificationToken(): ?string
    {
        return $this->verificationToken;
    }

    public function setVerificationToken(?string $verificationToken): void
    {
        $this->verificationToken = $verificationToken;
    }

    public function getLocale(): ?string
    {
        return $this->locale;
    }

    public function setLocale(string $locale): self
    {
        $this->locale = $locale;

        return $this;
    }
}
