<?php

declare(strict_types=1);

namespace App\Controller\Cms\Security;

use App\Controller\Cms\Abstract\BaseController;
use App\Entity\System\Administrator\Administrator;
use App\Form\System\Administrator\ChangePasswordType;
use App\Form\System\Administrator\ManageType;
use App\Form\System\Administrator\SettingsType;
use App\Table\System\AdministratorTable;
use App\Traits\Route\CmsAction;
use Scheb\TwoFactorBundle\Model\Google\TwoFactorInterface;
use Scheb\TwoFactorBundle\Security\TwoFactor\Provider\Google\GoogleAuthenticatorInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Symfony\Component\Routing\Annotation\Route;
use Endroid\QrCode\Builder\Builder;
use Endroid\QrCode\Encoding\Encoding;
use Endroid\QrCode\ErrorCorrectionLevel\ErrorCorrectionLevelHigh;
use Endroid\QrCode\RoundBlockSizeMode\RoundBlockSizeModeMargin;
use Endroid\QrCode\Writer\PngWriter;

#[Route('/administrator', name: 'administrator_')]
final class AdministratorController extends BaseController
{
    use CmsAction;

    final public const KEY = 'administrators';

    final public const PREFIX = 'administrator_';

    final public const ENTITY = Administrator::class;

    final public const FORM = ManageType::class;

    final public const TABLE = AdministratorTable::class;


    #[Route('/{id}/password-change', name: 'change_password', methods: ['GET', 'POST'])]
    public function passwordChange(
        Request       $request,
        Administrator $administrator
    ): Response
    {

        $form = $this->createForm(ChangePasswordType::class, $administrator, ['locale' => $request->getLocale()]);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $this->entityManager->flush();
            return $this->redirectToRoute($this->prefix . 'list', [], Response::HTTP_SEE_OTHER);
        }

        return $this->render('system/common/manage.html.twig', [
            'entity' => $administrator,
            'meta' => ['name' => $this->name, 'route_prefix' => $this->prefix],
            'form' => $form,
        ]);
    }

    #[Route('/settings', name: 'settings', methods: ['GET', 'POST'])]
    public function profileSettingsAction(Request $request): Response
    {
        $user = $this->getUser();
        $form = $this->createForm(SettingsType::class, $user, ['locale' => $request->getLocale()]);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $this->entityManager->flush();
            return $this->redirectToRoute($this->prefix . 'settings', [], Response::HTTP_SEE_OTHER);
        }

        return $this->render('system/common/manage.html.twig', [
            'entity' => $user,
            'meta' => ['name' => $this->name, 'route_prefix' => $this->prefix],
            'form' => $form,
        ]);
    }


    #[Route('/qr-code', name: 'qr_code', methods: ['GET'])]
    public function getQrCodeAction(
        Request $request,
        GoogleAuthenticatorInterface $googleAuthenticator
    ): JsonResponse
    {
        $user = $this->getUser();
        if (!($user instanceof TwoFactorInterface)) {
            throw new NotFoundHttpException('Cannot display QR code');
        }

        return new JsonResponse([
          'qrCode' => $this->displayQrCode($googleAuthenticator->getQRContent($user)),
          'token' => $user->getGoogleAuthenticatorSecret()
        ]);

    }

    private function displayQrCode(string $qrCodeContent): string
    {
        $result = Builder::create()
            ->writer(new PngWriter())
            ->writerOptions([])
            ->data($qrCodeContent)
            ->encoding(new Encoding('UTF-8'))
            ->errorCorrectionLevel(new ErrorCorrectionLevelHigh())
            ->size(400)
            ->margin(0)
            ->roundBlockSizeMode(new RoundBlockSizeModeMargin())
            ->build();

        return $result->getDataUri();
    }

}
