<?php

declare(strict_types=1);

namespace App\Command\Portal\User;

use App\Enum\User\UserDeletionReason;
use App\Event\User\UserDeletedEvent;
use App\Repository\User\UserRepository;
use DateTime;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;
use Symfony\Component\Messenger\Attribute\AsMessageHandler;

#[AsMessageHandler(handles: DeleteUser::class, bus: 'command.bus')]
final readonly class DeleteUserHandler
{
    public function __construct(
        private UserRepository           $userRepository,
        private EventDispatcherInterface $eventDispatcher
    )
    {
    }

    public function __invoke(DeleteUser $deleteUser): void
    {
        $user = $this->userRepository->findById($deleteUser->getId()) ?? throw new BadRequestHttpException();

        $userDeletionReason = $deleteUser->getReason();

        $user->setDeletedAt(new DateTime());
        $user->setDeletionReason($userDeletionReason);
        $user->setDeletionOtherReason($userDeletionReason === UserDeletionReason::OTHER ? $deleteUser->getOtherReason() : null);

        $this->userRepository->add($user, true);

        $this->eventDispatcher->dispatch(new UserDeletedEvent($user));
    }
}
